



/************************************************************************************************************
 **                                                                                                        **
 **       Examples1 :  Test ET-MINI MCP23017 Input(SW)/Output(LED) + INTB Of GPB                           **     
 **                     For "ET-BASE PIC8722 (ICD2)"  Board                                                **               
 **                                                                                                        ** 
 **                 :  Interface I2C (Use I/O Pin)                                                         ** 
 **                                                                                                        **
 ************************************************************************************************************
 **                                                                                                        **
 ** Target MCU        : PIC18F8722                                                                         **
 **                   : X-TAL : 10 MHz                                                                     **
 **                   : CPU Clock = 40 MHz (x4)                                                            **                                                                  *
 ** Editor-Compiler   : CCS Version 4.124                                                                  **
 **                                                                                                        ** 
 ** Create By         : Mr. Sittiphol Yooyod (WWW.ETT.CO.TH)                                               **
 ** Last Update       : 21/October/2013                                                                    **
 **                                                                                                        **
 ** ET-MINI MCP23017 :  Set Jumper PULL-UP SDA,SCL,RES to ENA                                              **
 **                     Set Jumper ADDRESS A2,A1,A0 = 0                                                    **
 **                                                                                                        **      
 ** Control MCP23017 : Interface I2C(ID:0100 000X = 0x40(Wr),0x41(Rd) By Set A2,A1,A0=0 )                  **  
 **                    Set Register IOCON : Use Address BANK0(BANK=0),Byte Mode(SEQOP=1)                   **
 **                    and Enable Address PIN A0,A1,A2                                                     **
 **                                                                                                        **
 **                                                                                                        **
 **                                                                                                        **
 ** I2C Interface  :   <MCU PIC18F8722>                     <MINI-MCP23017 >                               ** 
 **                                                                                                        **
 **                     RC3 (SCK-Out)      Connect            SCL                                          **
 **                     RC4 (SDA-I/O)      Connect            SDA(DATA)                                    **
 **                                                                                                        **
 **                                                                                                        **
 **                                                                                                        **
 **                                        5V                                                              **
 **                                       --+--                                                            **
 **                                         |                                                              **
 **                                         |              ID = 0x40(WR),41(RD)                            **   
 **                                      +--+--+-----+                                         5V          **
 **                                      |     |     |        MINI-MCP23017                   --+--        **
 **                                      |     |     |   +------------------+                   |          **
 **       5V                             \     \     |   |                  |                   |          **
 **      --+--                    4k7x2  /     /     |   |                  |     LED0    10k   |          **
 **        |                             \     \     +---+VCC          GPA0 +-----|<|---/\/\/\--+          **
 **        |           MCU PIC18F8722    /     /         |               .  |                   |          **
 **        |           +------------+    |     |         |               .  |                   |          **
 **        +-----------+            |    |     |         |               .  |     LED7    10k   |          **
 **                    |            |SCK |     |         |             GPA7 +-----|<|---/\/\/\--+          ** 
 **                    |        RC3 |----+-----|-------->|SCL               |                              **
 **                    |            |          |         |                  |                              **
 **                    |            |SDA       |         |                  |                              **
 **                    |        RC4 |<---------+-------->|SDA               |                              **
 **                    |            |                    |                  |                              **
 **                    |            |                    |                  |      SW0 (Set FullUp GPB)    **
 **              +-----+            |                   -|INTA         GPB0 +------/ ----------+           **
 **              |     |            |                    |               .  |                  |           **
 **              |     |            | Signal INT.GPB<----|INTB           .  |                  |           **                     
 **              |     |            |     active '0'     |               .  |      SW7         |           **
 **            --+--   |            |                    |             GPB7 +------/ ----------+           ** 
 **             ---    +------------+               +----+ GND              |                  |           **
 **              -                                  |    |                  |                  |           **
 **                                                 |    |    A2  A1  A0    |                  |           **
 **                                                 |    +----+---+---+-----+                --+--         **
 **                                               --+--       |   |   |                       ---          **
 **                                                ---        |   |   |                        -           **
 **                                                 -         +---+---+                                    **
 **                                                               |                                        **
 **                                                             --+--                                      **
 **                                                              ---                                       **
 **                                                               -                                        **
 **                                                                                                        **
 ************************************************************************************************************/


#include <18F8722.h>
#include <stdio.h>

#fuses H4,NOLVP,NOWDT,NOPROTECT,NOSTVREN         //Setup MCU
#use delay (clock = 40000000)                    //delay Clock = (ms)

#use fast_io(ALL)  //Stop Set direction Auto for All Port  by Compiler  



/*****************************************************************
 **     Define Register Address (IOCON.BANK=0 [Default])        **
 *****************************************************************/

 #define IODIRA     0x00     //Set Direction GPA : 1= Input ,0=Output
 #define IODIRB     0x01     //Set Direction GPB

 #define IOCON      0x0A     //Set Config Reg.

 #define GPPUA      0x0C     //Set Pullup Input PA  : 1=Enable PullUp ,0=Disable PullUp
 #define GPPUB      0x0D     //Set Pullup Input PB

 #define GPIOA      0x12     //IN/OUT DATA PORTA
 #define GPIOB      0x13     //IN/OUT DATA PORTB
 
 #define GPINTENA   0x04     //Reg. Select PIN Enable Interrupt-On-Chang For GPA
 #define GPINTENB   0x05     //Reg. Select PIN Enable Interrupt-On-Chang For GPB
  
 #define DEFVALA    0x06     //Reg. Select Value-bit Compare for Interrupt-On-Chang  In GPA  
 #define DEFVALB    0x07     //Reg. Select Value-bit Compare for Interrupt-On-Chang  In GPB  
 
 #define INTCONA    0x08     //Reg. Select Compare PIN GPA current with vale Reg.DEFVALA or Previous Value PIN GPA   
 #define INTCONB    0x09     //Reg. Select Compare PIN GPB current with vale Reg.DEFVALA or Previous Value PIN GPB



//************** Set I2C Control Pin *******************

#define SCL_HI()     output_high(PIN_C3) ;         //RC3 = ENA:1
#define SCL_LO()     output_low(PIN_C3)  ;         //RC3 = ENA:0

#define SDA_HI()     output_high(PIN_C4) ;         //RC4 = ENA:1
#define SDA_LO()     output_low(PIN_C4)  ;         //RC4 = ENA:0

#define SDA_IN()     set_tris_C(0x10)    ;         //Set RC4 = SDA: Input
#define SDA_OUT()    set_tris_C(0x00)    ;         //Set RC4 = SDA: Output
 



 /***************************************************************
  **       Define Control Byte Address for MCP23017            **
  **     When A0,A1,A2 = 0     (0100 A2A1A0 X)                 **
  ***************************************************************/
   
 #define  CTRL_Byte_WR   0x40  
 #define  CTRL_Byte_RD   0x41 


/****************************************************************
 **                     Function Initial Port                  **
 ****************************************************************/

 void init_port(void)
  {

   set_tris_C(0x00)   ;   //Set RC3(SCK) = output  ,RC4(SDA) = Output

  }


/********************************************************************************
 **                                                                            **
 **                         Function I2C Interface                             **
 **                  - 8 bit Interface , Sent Bit 7 First bit                  **
 **                                                                            **
 ********************************************************************************/


 /**********************************************
  **                 I2C Start                **
  **********************************************/

void I2C_Start(void)
{
   
   SDA_OUT()              ;  //Set RC4(SDA) = output  
 
   SDA_HI()               ;   
   SCL_HI()               ;
   SDA_LO()               ;    
   SCL_LO()               ;
}



 /*********************************************
  **                 I2C Stop                **
  *********************************************/

void I2C_Stop(void)
{

   SDA_OUT()             ;  //Set RC4(SDA)=Output      
  
   SDA_LO()              ;  
   SCL_HI()              ;     
   SDA_HI()              ;
   
}


/*********************************************
 **         I2C Write  Data 8 bit           **
 **                                         **
 *********************************************
 ** Parameter : dat = Data for write 1Byte  **                                  
 *********************************************/


void I2C_WrByte(unsigned int8 dat)              
{
   unsigned int8 loop ;      


   for(loop=0;loop<8;loop++)                 //Loop Write data 8 Bit 
    {                   
      //--------- write Data(SDA:RC4) ---------

     SDA_OUT() ;                            //Set PD1=SDA= Output  

     if((dat & 0x80)== 0x80)                //Check data bit7 is 0 or 1
     {
        SDA_HI()                         ;  //Sent '1' Out Pin SDA(RC4)
     }
     else
     {
       SDA_LO()                          ;  //Sent '0' Out Pin SDA(RC4)
     }
     dat = dat<<1                        ;  //Shift data Next Bit 

     SCL_HI()                            ;  //Clock HI  
     delay_us(10)                        ;  //Delay Clock 10us                  
     SCL_LO()                            ;  //Clock LO       
     delay_us(10)                        ;  //Delay Clock 10us  

   } 

   SDA_IN()                              ;  //Set PD1=Input 

   SCL_HI()                              ;  //Start ACK Clock
   while(input(PIN_C4)== 0x01){;}           //Check bit ACK:RC4 = 0 Out Loop
   SCL_LO()                              ;  //End ACK Clock
             


}


 
/************************************************
 **        I2C Read  Data 8 bit                **
 **                                            **           
 ************************************************/

 signed int8 I2C_RdByte(void)              
 { 
   signed int8 loop,result=0 ;      

   for(loop=0;loop<8;loop++)           //Loop Read data 8-Bit
    {
      result <<= 1                 ;   //Shift Result Save (MSB <- LSB)
    
      SDA_IN()                     ;   //Set RC4=SDA= Input 
        
      SCL_HI()                     ;   //Strobe SCK Read SDA
             
     if(input(PIN_C4) == 0x01)         //Check Data Read Pin SDA(RC4) is '0' or '1'
        result |= 0x01             ;   //If Read Pin SDA is '1' Give Save Bit Data = 1

      SCL_LO()                     ;   //Next Bit Read

    }
    
  return result                    ;

}



/******************************************************************************* 
 **               Function I2C Write MCP23017 (Byte-Mode)  1 Sequence         **
 ** Parameter :                                                               **
 **            Control Byte Write MCP23017 = 0x40 (A2,A1,A0=0)                **   
 **            addr  = Register Address for write(1Byte)                      **
 **            dat   = Data to MCP23017                                       **    
 *******************************************************************************/
 
void I2C_WrMCP23017 (unsigned int8 addr,unsigned int8 dat)
{ 
  
   I2C_Start() ;

//---------- Sent Slave address for Write ---------------

   I2C_WrByte(CTRL_Byte_WR)      ;  //Send ID Code MCP32017+Write (0100 000+0) = 0x40
 
//------------ Sent address Register --------------
 
   I2C_WrByte(addr)              ;   //Send Address Reg(0x00-0x15). to MCP23017

//------------- Sent data to MCP23017 ---------------

   I2C_WrByte(dat)               ;   //Send Data Control to MCP23017
  
   I2C_Stop()                    ;
 
 
}



/******************************************************************************
 **              Function I2C Read MCP32017 (Byte-Mode) 1 Sequence           **
 **                                                                          **
 ** Parameter :                                                              **
 **             Control Byte Read MCP23017 = 0x41 (A2,A1,A0=0)               **
 **             addr = Register Address for Read (1Byte)                     **
 ******************************************************************************/

 

 signed int8 I2C_RdMCP23017(unsigned int8 addr)
  {
    signed int8 Rdx  ;
   
 //-------------- Sent Status Start for write -----------

   I2C_Start();

//------------- Sent Slave Address for Write------------   
  
   I2C_WrByte(CTRL_Byte_WR)    ;       //Send ID Code MCP23017,Write (0100 000+0) = 0x40

//------------ Sent address Register --------------

   I2C_WrByte(addr)            ;        //Send Address Reg. for Read MCP32017
 
//---------- New Sent Status Start For Read ----------

   I2C_Start()                 ;

//-------------Sent Slave address for Read ------------

   I2C_WrByte(CTRL_Byte_RD)    ;       //Send ID Code MCP23017 ,Write (0100 000+1) = 0x41

//------------- Read data 1 Byte ------------
   
   Rdx = I2C_RdByte()          ;               
  

//---------- Sent Status Stop Red ---------------
  
   I2C_Stop()                  ; 
 
   return Rdx                  ;        //Return Data

  }




/*****************************************
 **     Function Initial MCP23017       **
 *****************************************/

 void init_MCP23017(void)
  {
   
    //---------------- Setup I/O Port --------------

    I2C_WrMCP23017(IOCON,0x28)    ;  //Config BANK0,Byte Mode,Enable MCP23017 AddressPin,INTB associated PGB

    I2C_WrMCP23017(IODIRA,0x00)   ;  //Set GPA = Output For LED
    I2C_WrMCP23017(IODIRB,0xFF)   ;  //Set GPB = Input For SW
   
    I2C_WrMCP23017(GPPUB,0xFF)    ;  //PullUp GPB Enable 


    //---------- Set Up Output Interrupt INTB ------------
 
   I2C_WrMCP23017(GPINTENB,0xFF)   ; // Enable GPB[0..7] pin For Interrupt-On-Chang
    
   I2C_WrMCP23017(DEFVALB,0x0F)    ; // Set Vale Compare to '1',Use 4 bit Low Compare with GPB[0..3] 
  
   I2C_WrMCP23017(INTCONB,0x0F)    ; // Set Bit GPB[0..3] Compare with Value Reg.DEFVALB ,And GPB[4..7] Compare with previous GPB[4..7] PIN

 
  }




/* ################################################################
   ##                                                            ##
   ##                        Main Program                        ##
   ##                                                            ##
   ################################################################ */
 
void main(void)
   {  
     int8 sw = 0 ;

     init_port()      ;
     init_MCP23017()  ;


     while(true)
     {
       
         sw = I2C_RdMCP23017(GPIOB)           ;   //Read Data from SW. GPB
         
       if(sw != 0xFF)                             //Check Press SW Bit=0
         {
           I2C_WrMCP23017(GPIOA,sw)           ;   //On LED(Logic 0) bit at press sw.
         
           do                                     //Protect Press SW.Rebeat or Double
           {
             sw = I2C_RdMCP23017(GPIOB)       ;   //Read Data from SW. GPB

           }while(sw != 0xFF)                 ;   //Check Release SW. 

         } 
        else  //Not Press SW. (0xFF)
         {
            I2C_WrMCP23017(GPIOA,0xFF)        ;   //Clear LED OFF  (LED Active Low)
         }     

          delay_us(500)                       ;   //delay for Bout SW. 
     }

   
   }
 


